<?php

class ControllerExtensionFeedMergadoMarketingPack extends Controller {

  const MIN_IMAGE_HEIGHT = 50;
  const MIN_IMAGE_WIDTH = 50;
  const MAX_IMAGE_HEIGHT = 4096;
  const MAX_IMAGE_WIDTH = 4096;
  const MAX_IMAGE_FILESIZE = 2000000; // in Bytes
  const VERSION = '1.0.0';

  private $extension_path = 'feed/';
  private $extension_name = 'mergado_marketing_pack'; 
  private $extension_fullname = 'feed_mergado_marketing_pack';
  private $extension_type = 'feed';

  private $user_currency;
  private $default_store_currency;
  private $zone_id;
  private $country_id;
  private $base_url;
  private $is_https;
  private $has_seo_url;
  private $stores = array();
  private $model = array();
  private $attrs = array();
  private $time = 0;
  private $extension_status;
  private $feed_hash;
  
  public function index() {

    $this->time = new DateTime('NOW');;

    //load models
    $this->loadModels();

    //set logger
    define('MERGADO_LOGGER_ENABLED', (int) $this->config->get($this->extension_fullname . '_logs'));

    //GET atributes
    $this->attrs = array();
    foreach($_GET as $key=>$value) {
      $this->attrs[$key] = htmlspecialchars($value);
    }

    //check security token
    if(!isset($this->attrs['h'])) {
      $this->model['logger']->log('invalid_token', array('token' => $this->attrs['h']), $this->attrs['s'], $this->attrs['l'], $this->attrs['c']);
      echo -2;
      exit;
    } elseif ( isset($this->attrs['a']) && isset($this->attrs['s']) && isset($this->attrs['l']) && isset($this->attrs['c']) ) { //check product feed hash token
      if( $this->attrs['a'] == 'pf' && md5($this->model['main']->getMergadoToken() . '_pf_' . $this->attrs['s'] . '_' . $this->attrs['l'] . '_' . $this->attrs['c']) != $this->attrs['h']) {
        echo -2;
        exit;
      }
    } elseif ( isset($this->attrs['a']) && isset($this->attrs['s']) ) { //check store log hash token
      if( $this->attrs['a'] == 'log' && md5($this->model['main']->getMergadoToken() . '_log_' . $this->attrs['s']) != $this->attrs['h']) {
        echo -2;
        exit;
      }
    } elseif ( isset($this->attrs['a']) ) { //check full log hash token
      if( $this->attrs['a'] == 'log' && $this->model['main']->getMergadoToken() != $this->attrs['h']) {
        echo -2;
        exit;
      }
    }

    //extension status
    $this->extension_status = $this->config->get($this->extension_fullname . '_status');

    try {
      // choose action based on URL parameter "a" (action)
      if(isset($this->attrs['a']) && $this->attrs['a'] == 'log') {          // log
          $this->showLog(isset($this->attrs['s']) ? $this->attrs['s'] : '');
      } elseif ($this->extension_status && isset($this->attrs['a']) && $this->attrs['a'] == 'pf') {    // product feed
          $this->createProductFeed();
      } elseif ($this->extension_status && isset($this->attrs['a']) && $this->attrs['a'] == 'cf') {    // category feed
          // Not supported in current version
          $this->model['logger']->log('Error: Category feed is not supported in this version. parameter a ==', $this->attrs['a']);
          exit;
      } elseif ($this->extension_status && isset($this->attrs['a']) && $this->attrs['a'] == 'af') {    // analytics feed
          // Not supported in current version
          $this->model['logger']->log('Error: Analytics feed is not supported in this version. parameter a ==', $this->attrs['a']);
          exit;
      } else {      
          
          if($this->extension_status) {
            // missing or incorrect params
            $this->model['logger']->log('error', 'missing_or_incorrect_params');
          } else {
            $this->model['logger']->log('warning', 'extension_is_disabled');
          }
          echo -1;
          exit;    
      }
    } catch(Exception $e) {
      $this->model['logger']->log('exception', $e->getMessage()); 
    }
    
  }

  private function createProductFeed() {
    // Start timer
    $this->model['logger']->log('product_generator_start', array(), $this->attrs['s'], $this->attrs['l'], $this->attrs['c']);

    //load stores
    $this->stores = $this->model['store']->getStores();

    // check if store id passed through parameter is valid:
    if (isset($this->attrs['s']))
    {
      if ($this->attrs['s'] != 0){
        // Check if the store_id is valid (we do not need to check store_id=0, since that is default one):
        $store_id_found = false;
        foreach($this->stores as $store){
          if (in_array($this->attrs['s'], $store)){
            $store_id_found = true;
          }
        }
        if (!$store_id_found){
          $this->model['logger']->log('catalog/controller - Error: Invalid store_id: ', $this->attrs['s'], $this->attrs['l'], $this->attrs['c']);
          exit;
        }
      }
    } else {
      $this->model['logger']->log('catalog/controller - Error: Failed reading of store_id from URL.', '', $this->attrs['s'], $this->attrs['l'], $this->attrs['c']);
      exit;
    }
    
    // read parameter c (currency) to find out the desired currency of export:
    if (isset($this->attrs['c']))
    {
      $this->user_currency = strtoupper($this->attrs['c']);
      // TODO: Add a check if the currency code supplied via URL is valid
    } else {
      $this->model['logger']->log('Error: Failed reading of currency from URL.','Parameter: c', $this->attrs['s'], $this->attrs['l']);
      exit;
    }

    //load global settings
    $this->is_https = $this->config->get('config_secure');
    $this->has_seo_url = $this->config->get('config_seo_url');
    $this->base_url = $this->is_https ? HTTPS_SERVER : HTTP_SERVER;
    $this->default_store_currency = $this->config->get('config_currency');

    // Change the language config so that model returns product data in correct language:
    if(isset($this->attrs['l'])) {
      $languages = $this->model['language']->getLanguages();
      $this->config->set('config_language_id', $languages[$this->attrs['l']]['language_id']);
    } else {
      $this->model['logger']->log('Error: Failed reading of language from URL.','Parameter: l', $this->attrs['s'], $this->attrs['l'], $this->attrs['c']);
      exit;
    }

    $products = $this->model['product']->getProducts();

    $output  = '<?xml version="1.0" encoding="utf-8"?>';
    $output .= '<CHANNEL xmlns="http://www.mergado.com/ns/1.5">';
    $output .= '<GENERATOR>mergado.opencart.marketingpack.' . str_replace('.', '-', self::VERSION) . '</GENERATOR>';
  
    foreach ($products as $product) {
      // If special price is set, use it instead of regular price:
      $options_data = $this->generate_options_data($product['product_id'], $product['name']);
      
      foreach ($options_data as $option_data) {
        $output .= '<ITEM>';
        $output .= '<ITEM_ID>' . $option_data['id'] . '</ITEM_ID>';
        $output .= '<NAME_EXACT><![CDATA[' . $option_data['name'] . ']]></NAME_EXACT>';
        $output .= '<DESCRIPTION><![CDATA[' . $this->plainText($product['description']) . ']]></DESCRIPTION>';
        $output .= '<URL><![CDATA[' . $this->replaceStoreURL($this->createURL($product['product_id'], "")) . ']]></URL>';
    
        //main image
        if($product['image']){
          $output .= '<IMAGE><![CDATA[' . $this->getImageURL($product['image']) . ']]></IMAGE>';
        }else{
          $output .= '<IMAGE></IMAGE>';
        }

        //alternative images
        $images = $this->model['product']->getProductImages($product['product_id']);
        foreach($images as $img) {
          $output .= '<IMAGE_ALTERNATIVE><![CDATA[' . $this->getImageURL($img['image']) . ']]></IMAGE_ALTERNATIVE>';
        }

        //categories
        $categories = $this->model['product']->getCategories($product['product_id']);
        foreach($categories as $cat) {
          $output .= '<CATEGORY><![CDATA[' . $this->getFullCategory($cat['category_id']) . ']]></CATEGORY>';
        }

        $output .= '<PRICE>' . $this->getPrice($product['price'], $product['special'], $option_data['delta_price']) . '</PRICE>';
        $output .= '<PRICE_VAT>' . $this->getPrice($product['price'], $product['special'], $option_data['delta_price'], $product['tax_class_id']) . '</PRICE_VAT>';
        $output .= '<CURRENCY>' . $this->user_currency . '</CURRENCY>';
        $output .= '<VAT>' . $this->getTaxRate($product['tax_class_id'] ) . '</VAT>';
        $output .= '<EAN>' . $product['ean'] . '</EAN>';
        $output .= '<ISBN>' . $product['isbn'] . '</ISBN>';
        $output .= '<PRODUCTNO><![CDATA[' . $product['model'] . ']]></PRODUCTNO>';
        $output .= '<AVAILABILITY>' . $this->getAvailability($product['quantity'], $product['stock_status']) . '</AVAILABILITY>';
        $output .= '<DELIVERY_DAYS>' . $this->getDeliveryDays($product['quantity'], $product['stock_status'], $product['date_available']) . '</DELIVERY_DAYS>';
        $output .= '<SHIPPING_SIZE><![CDATA[' . $product['length'] . ' x ' . $product['width'] . ' x ' . $product['height'] . ' '. $this->model['main']->getLengthClass($product['length_class_id'], $languages[$this->attrs['l']]['language_id'])['unit'] . ']]></SHIPPING_SIZE>';
        $output .= '<SHIPPING_WEIGHT><![CDATA[' . ($product['weight'] + $option_data['delta_weight']) . ' '. $this->model['main']->getWeightClass($product['weight_class_id'], $languages[$this->attrs['l']]['language_id'])['unit'] . ']]></SHIPPING_WEIGHT>';
        $output .= '<PRODUCER><![CDATA[' . $product['manufacturer'] . ']]></PRODUCER>';
        $output .= '<ITEMGROUP_ID>' . $product['product_id'] . '</ITEMGROUP_ID>';
        
        $output .= '</ITEM>';
      }
    }
    $output .= '</CHANNEL>';
    
    $this->feed_hash = md5((isset($this->attrs['a']) ? $this->attrs['a'] : '') . '_' . (isset($this->attrs['s']) ? $this->attrs['s'] : '') . '_' . (isset($this->attrs['l']) ? $this->attrs['l'] : '') . '_' . (isset($this->attrs['c']) ? $this->attrs['c'] : ''));
    $filename = 'pf_' . $this->feed_hash;
    $this->createFeed($filename, $output);
    exit;
  }

  private function generate_options_data($product_id, $name) {
    $options = $this->model['product']->getProductOptions($product_id);
    $output_array = array();

    if(!empty($options)){
      $input_array = array();
      $input_array[0] = array("id"=>$product_id, "name"=>$name, "delta_price"=>0, "delta_weight"=>0);
    
      foreach ($options as $option) {
        if(!empty($option['product_option_value'])){
          $output_array = array();
          foreach ($input_array as $input_array_member) {
            foreach ($option['product_option_value'] as $product_option_value_iterator) {
              $temp_id = $input_array_member['id'] . '-' . $option['product_option_id'] . '_' . $product_option_value_iterator['product_option_value_id'];
              
              $temp_name = $input_array_member['name'] . ' ' . $product_option_value_iterator['name'];
              
              if($product_option_value_iterator['price_prefix'] == "+") {
                $temp_price = $input_array_member['delta_price'] + $product_option_value_iterator['price'];
              } else {
                $temp_price = $input_array_member['delta_price'] - $product_option_value_iterator['price'];
              }

              if($product_option_value_iterator['weight_prefix'] == "+") {
                $temp_weight = $input_array_member['delta_weight'] + $product_option_value_iterator['weight'];
              } else {
                $temp_weight = $input_array_member['delta_weight'] - $product_option_value_iterator['weight'];
              }

              array_push($output_array, array("id"=>$temp_id, "name"=>$temp_name, "delta_price"=>$temp_price, "delta_weight"=>$temp_weight));
            }
          }
          $input_array = $output_array;
        }
      }
      if(empty($output_array)){
        // in case there are no options just copy data to output_array so that we can iterate just through that and keep the code common
        $output_array[0] = array("id"=>$product_id, "name"=>$name, "delta_price"=>0, "delta_weight"=>0);
      }
    } else {
      // in case there are no options just copy data to output_array so that we can iterate just through that and keep the code common
      $output_array[0] = array("id"=>$product_id, "name"=>$name, "delta_price"=>0, "delta_weight"=>0);
    }
    
    return $output_array;
  }

  private function getImageURL($image_path) {
    $ret_val = $this->hasImageValidSize(DIR_IMAGE. $image_path);
    if($ret_val == 1){
      return $this->replaceStoreURL($this->base_url . 'image/' . $image_path);
    }elseif($ret_val == -1){ // image smaller than minimum
      return $this->replaceStoreURL($this->model['image']->resize( $image_path, self::MIN_IMAGE_WIDTH, self::MIN_IMAGE_HEIGHT));
    }elseif($ret_val == -2){ // image bigger than maximum
      return $this->replaceStoreURL($this->model['image']->resize( $image_path, self::MAX_IMAGE_WIDTH, self::MAX_IMAGE_HEIGHT));
    }else{ //should get where when $ret_val=3; image size on disk bigger than maximum
      return '';
    }
  }

  private function loadModels() {

    $this->load->model('setting/setting');
    $this->load->model('catalog/product');
    $this->load->model('catalog/category');
    $this->load->model('tool/image');
    $this->load->model('localisation/zone');
    $this->load->model('localisation/country');
    $this->load->model('localisation/currency');
    $this->load->model('localisation/language');
    $this->load->model('setting/store');
    $this->load->model('extension/' . $this->extension_path . $this->extension_name);
    $this->load->model('extension/' . $this->extension_path . $this->extension_name . '_logger');
    $this->load->model('extension/' . $this->extension_path . $this->extension_name . '_generator');

    //custom models
    $model_name = "model_extension_" . $this->extension_type . "_" . $this->extension_name;
    $model_logger = "model_extension_" . $this->extension_type . "_" . $this->extension_name . '_logger';
    $model_generator = "model_extension_" . $this->extension_type . "_" . $this->extension_name . '_generator';

    $this->model['settings'] = $this->model_settings_setting;
    $this->model['product'] = $this->model_catalog_product;
    $this->model['category'] = $this->model_catalog_category;
    $this->model['image'] = $this->model_tool_image;
    $this->model['zone'] = $this->model_localisation_zone;
    $this->model['country'] = $this->model_localisation_country;
    $this->model['currency'] = $this->model_localisation_currency;
    $this->model['language'] = $this->model_localisation_language;
    $this->model['store'] = $this->model_setting_store;
    $this->model['main'] = $this->{$model_name};
    $this->model['logger'] = $this->{$model_logger};
    $this->model['generator'] = $this->{$model_generator};

  }

  public function createURL($product_id, $url_param = ""){
    $url = $this->url->link('product/product', 'product_id=' . $product_id, $this->is_https);
    if($this->has_seo_url){
      return  str_replace("&amp;","&",$url) . ($url_param!="" ? "?" . str_replace("?","",$url_param) : "");
    }else{
      return  str_replace("&amp;","&",$url) . ($url_param!="" ? "&" .str_replace("?","",$url_param) : "");
    }
  }

  public function replaceStoreURL($orig_url){
    // For not default stores replace domain with what is stored in store url for that store,
    // as it is not done automatically in older versions of OpenCart
    if ($this->store_id != 0){
      // Find the index of the store with given $this->store_id in $this->stores
      foreach($this->stores as $index => $store){
        if (in_array($this->store_id, $store)){
          break;
        }
      }

      // parse and replace domain part of orig_url for store url or store's ssl url:
      $elements = explode('/', $orig_url);
      if ($this->stores[$index]['ssl'] && $this->is_https) {
        $new_url = str_replace($elements[0] . '//' . $elements[2] . '/', $this->stores[$index]['ssl'], $orig_url);
      } else {
        $new_url = str_replace($elements[0] . '//' . $elements[2] . '/', $this->stores[$index]['url'], $orig_url);
      }

      return $new_url;
    } else {
      return $orig_url;
    }
  }

  private function getFullCategory($category_id) {
    $categories = array();

    $category_data = $this->model['category']->getCategory($category_id);
    $categories[] = $category_data['name'];

    while( isset($category_data['parent_id']) && $category_data['parent_id'] != 0 ){
      $category_data = $this->model['category']->getCategory($category_data['parent_id']);
      $categories[] = $category_data['name'];
    }

    return implode(' / ', array_reverse($categories));
  }

  private function getPrice($price , $special, $delta_price=0, $tax_class_id = null) {

    $this->getCountyId();
    $this->getZoneId();

    $this->tax->setStoreAddress($this->country_id, $this->zone_id);
    $this->tax->setPaymentAddress($this->country_id, $this->zone_id);
    $this->tax->setShippingAddress($this->country_id, $this->zone_id);

    if((float) $special){
      $computed_price = $this->currency->convert($special + $delta_price, $this->default_store_currency, $this->user_currency);
    } else {
      $computed_price = $this->currency->convert($price + $delta_price, $this->default_store_currency, $this->user_currency);
    }

    if(!is_null($tax_class_id)) {
      return number_format($this->tax->calculate($computed_price, $tax_class_id), $this->currency->getDecimalPlace($this->user_currency));
    } else {
      return number_format($computed_price, $this->currency->getDecimalPlace($this->user_currency));
    }
  }

  private function getTaxRate($tax_class_id) {
    $rates = $this->tax->getRates(0, $tax_class_id);
    $vat = 0;
    foreach($rates as $r) {
      if($r['type'] == 'P') {
        $vat += $r['rate'];
      }
    }

    return $vat;
  }

  private function getAvailability($quantity, $order_status) {
    if($quantity > 0) {
      return 'in stock';
    } else {
      //treba doriesit pripady pre stav pre-order
      return 'out of stock';
    }
  }

  private function getDeliveryDays($quantity, $order_status, $date_available) {
    if($quantity > 0) {
      return 0;
    } else {

      $now = new DateTime();
      $dt_available = new DateTime($date_available);
      $interval = $now->diff($dt_available);
      
      //treba doriesit pripady pre stav pre-order

      if($interval->d > 0) {
        return $interval->d;
      } 
      return '';
    }
  }

  private function getCountyId(){
    $countries = $this->model['country']->getCountries();
    if(!empty($countries)){
      foreach($countries as $c){
        if(strtolower($c['iso_code_2']) == explode('-',$this->attrs['l'])[1]){
            $this->country_id = $c['country_id'];
            return;
        }
      }
    }
  }

  private function getZoneId(){
    $zones = $this->model['zone']->getZonesByCountryId($this->country_id);
    if(!empty($zones)){
      foreach($zones as $z){
        $this->zone_id = $z['zone_id'];
        return;
      }
    }
  } 

  public function createFeed($feed_name, $data){
    $root_dir = DIR_APPLICATION.'../mergado/';

    //create root dir
    if(!file_exists($root_dir)) {
      mkdir($root_dir);
      chmod($root_dir, 0755);
    }
    
    $feed_path = $root_dir . $feed_name . '.xml';
    @file_put_contents($feed_path, $data);

    if(file_exists($feed_path)){
      chmod($feed_path, 0755);

      $final_time = new DateTime('NOW');

      $this->model['logger']->log('product_generator_finished', array(
        'status' => 'OK',
        'duration' => $final_time->diff($this->time)->s
      ), $this->attrs['s'], $this->attrs['l'], $this->attrs['c']);

      $this->model['generator']->update($this->feed_hash, 1);

      echo 1; 

    }else{
      $final_time = new DateTime('NOW');

      $this->model['logger']->log('product_generator_finished', array(
        'status' => 'ERR',
        'duration' => $final_time->diff($this->time)->s
      ), $this->attrs['s'], $this->attrs['l'], $this->attrs['c']);

      //update feed generator status
      $this->model['generator']->update($this->feed_hash, -1);

      echo -1; 
    }

    exit;
  }

	public function plainText($string){
    return strip_tags(html_entity_decode($string));
	}

  public function hasImageValidSize($path){
    if(file_exists($path)){
      list($width, $height) = @getimagesize($path);
      $filesize = filesize($path); // in bytes
      
      if( $width < self::MIN_IMAGE_WIDTH && $height < self::MIN_IMAGE_HEIGHT ){
        return -1; // image smaller than minimum
      }
      
      if( $width > self::MAX_IMAGE_WIDTH && $height > self::MAX_IMAGE_HEIGHT ){
        return -2; // image bigger than maximum
      }
      
      if( $filesize > self::MAX_IMAGE_FILESIZE ){
        return -3; // image size on disk bigger than maximum
      }
    }

    return 1;
  }

  public function showLog($store_id = '') {
    header('Content-Type: text/plain');

    $params = array();
    if($store_id != '') {
      $params = array('store_id' => array(
        'value' => $store_id, 
        'type' => 'int'
        )
      );
    }

    $logs = $this->model['logger']->getLogs($params);

    $output = '';
    foreach($logs as $log) {
      $output .= '[' . $log['log_date'] . '][store id: ' . $log['store_id'] . ($log['lang_code'] != '' ? ', lang: ' . $log['lang_code'] : '') . ($log['currency'] != '' ? ', currency: ' . $log['currency'] : '') . '] ' . $log['log_label'] . ' '. $log['log_msg'] . PHP_EOL;
    }

    echo $output;
    exit;
  }
}
